import SwiftUI

private struct Spinning3DRotationModifier: ViewModifier {
  @State private var angle: Double = 0
  let duration: Double
  let axis: (x: CGFloat, y: CGFloat, z: CGFloat)
  let anchor: UnitPoint
  let startAngle: Double
  let endAngle: Double

  func body(content: Content) -> some View {
    content
      .rotation3DEffect(.degrees(angle), axis: axis, anchor: anchor)
      .onAppear {
        angle = startAngle
        withAnimation(.linear(duration: duration).repeatForever(autoreverses: false)) {
          angle = endAngle
        }
      }
  }
}

extension View {
  func spinning3DRotation(
    duration: Double = 4,
    axis: (x: CGFloat, y: CGFloat, z: CGFloat) = (1, 1, 1),
    anchor: UnitPoint = .leading,
    startAngle: Double = 0,
    endAngle: Double = 360
  ) -> some View {
    modifier(Spinning3DRotationModifier(
      duration: duration,
      axis: axis,
      anchor: anchor,
      startAngle: startAngle,
      endAngle: endAngle
    ))
  }
}

struct ContentView: View {
  var body: some View {
    VStack {
      Image(systemName: "globe")
        .font(.system(size: 200))
        .foregroundStyle(.tint)
        .padding(.leading, 100)
        .spinning3DRotation()
    }
    .padding()
  }
}

#Preview {
  ContentView()
}

// MARK: Mario Geneau - 10/2/25 - v1

/// - Parameters:
///   - duration: The duration (in seconds) for a full rotation cycle. Default is 4 seconds.
///   - axis: The axis of rotation as a tuple of x, y, z. Default is (1,1,1).
///   - anchor: The anchor point for the rotation. Default is `.leading`.
///   - startAngle: The starting angle in degrees. Default is 0.
///   - endAngle: The target angle in degrees. Default is 360.
