import SwiftUI

// 4. In Swift, the extension keyword lets you add new functionality to an existing type.
extension Color {
  // 6. Custom initializer added to SwiftUI’s Color type via an extension.
  init(hex: String, alpha: Double = 1.0) {
    // 8.  One-liner that sanitizes and normalizes a hex color string before parsing it.
    let cleaned = hex.trimmingCharacters(in: CharacterSet.alphanumerics.inverted).uppercased()
    // 10. A single line that declares and initializes three variables in one go.
    var r: UInt64 = 255, g: UInt64 = 255, b: UInt64 = 255
    // 12. Optional parsing of a hex string into an integer.
    if let int = UInt64(cleaned, radix: 16) {
      // 14 to 16. Extracting the red, green, and blue components from a 24-bit hex color integer using bit shifting and masking.
      r = (int >> 16) & 0xFF
      g = (int >> 8) & 0xFF
      b = int & 0xFF
    }
    // 19. Begins the actual construction of a Color value inside your custom initializer.
    self.init(
      // 21. sRGB is specifying the color space used for the red/green/blue values you’re passing in.
      .sRGB,
      // 23 to 25. Convert your 8-bit RGB components into normalized Double values that SwiftUI’s Color initializer expects.
      red: Double(r) / 255.0,
      green: Double(g) / 255.0,
      blue: Double(b) / 255.0,
      // 27. Alpha simply forwards the caller’s alpha argument to the opacity parameter of Color’s initializer.
      opacity: alpha
    )
  }
}

// MARK: - Preview Demo

private struct ColorHexDemoView: View {
  var body: some View {
    VStack(spacing: 12) {
      HStack(spacing: 12) {
        colorSwatch(title: "#FF0000", color: Color(hex: "#FF0000"))
        colorSwatch(title: "#00FF00", color: Color(hex: "00FF00"))
        colorSwatch(title: "#0000FF", color: Color(hex: "0000FF"))
      }
      HStack(spacing: 12) {
        colorSwatch(title: "#1ABC9C", color: Color(hex: "#1ABC9C"))
        colorSwatch(title: "#F1C40F", color: Color(hex: "F1C40F"))
        colorSwatch(title: "#8E44AD 60%", color: Color(hex: "8E44AD", alpha: 0.6))
      }
    }
    .padding(16)
    .background(Color(.systemBackground))
  }

  // 53. @ViewBuilder is a SwiftUI result builder attribute that lets you write multiple view expressions inside a function and have them combined into a single some View result automatically.
  @ViewBuilder
  private func colorSwatch(title: String, color: Color) -> some View {
    VStack {
      RoundedRectangle(cornerRadius: 8)
        .fill(color)
        .frame(width: 64, height: 64)
        .overlay(
          RoundedRectangle(cornerRadius: 8)
            .stroke(Color.primary.opacity(0.1), lineWidth: 1)
        )
      Text(title)
        .font(.caption)
        .foregroundStyle(.secondary)
        .lineLimit(1)
        .minimumScaleFactor(0.7)
    }
  }
}

#Preview("Color(hex:) Demo") {
  ColorHexDemoView()
}

// MARK: Mario Geneau - 10/7/25 - v2
